"""
sr_v2_demo_baseline.py
=======================

This script implements the baseline V2 present‑act engine SR demo. It performs a
clean acceptance run by:

  * Declaring the governance checklist (non‑negotiable policies).
  * Setting defaults for the seed, gates (Θ, κ), per‑act bound b, norm, and
    allocation grid α.
  * Checking the ledger sector diagnostic using S†S + X†X = I for each α.
  * Implementing the per‑act bound and verifying no‑skip composition.
  * Simulating worldlines for each α in the grid to measure γ_meas and compare
    against γ_theory = 1/√(1‑α²).
  * Building baseline cone envelopes and demonstrating gate variations.
  * Checking re‑centering invariance by rotating the sequence of Δx values.
  * Auditing no‑skip factorization.
  * Optionally running a simple homeostatic allocation series.
  * Generating plots, CSV tables, JSON/YAML manifests, and a text report.

All outputs are written to the script's working directory (controlled by
`sr_out_dir`). The script is self‑contained and only uses standard libraries
(`json`, `yaml`), numpy, pandas, and matplotlib. No external data is required.

Usage: simply run `python sr_v2_demo_baseline.py` from within the repo root or
any directory. The default `sr_out_dir` is the current working directory.
"""

import json
import math
import os
import yaml

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt


def run_baseline(sr_out_dir=".", seed=20251020):
    """Run the baseline SR acceptance demo and write outputs to sr_out_dir.

    Parameters
    ----------
    sr_out_dir : str
        The directory in which to write plots, tables, manifests, and reports.
        If the directory does not exist, it will be created.
    seed : int
        The random seed used to seed numpy's RNG. This affects only tie‑break
        randomization, which is not used in the baseline run.
    """
    # Create output directory if it doesn't exist
    os.makedirs(sr_out_dir, exist_ok=True)

    # 0) Governance checklist (printed for auditing)
    governance_checklist = [
        "Roles on arrows only (L1/L2/L3 as arrow roles).",
        "No curves/pivots/UGM as weights; diagnostics only.",
        "No‑skip composition (long moves factor neighbour by neighbour).",
        "Gates are feasibility, not weights: Theta (act window), kappa (minimal outward part).",
        "Fit rule Φ(W,Q)=0 via lexicographic order: gates → minimize d_out → then d_in → then d_×; tie = fewest acts; seed only after feasibility.",
        "Ledger is diagnostic only; never fed back into fit/gates/bounds.",
        "Canonical units & invariant mapping c (we use c=1 unless overridden).",
    ]

    print("0) Governance checklist (declared):")
    for item in governance_checklist:
        print("  -", item)

    # 1) Defaults
    np.random.seed(seed)
    c = 1.0
    Theta_ladder = [1.0]
    kappa_ladder = [1.0]
    norm = "frobenius"
    alpha_grid = [0.0, 0.25, 0.5, 0.75, 0.9]
    n_acts = 1000
    dt = 1.0
    
    print("\n1) Defaults in use:")
    print(f"   seed = {seed}")
    print(f"   c = {c}")
    print(f"   Theta = {Theta_ladder}")
    print(f"   kappa = {kappa_ladder}")
    print(f"   allocation grid α = {alpha_grid}")
    print(f"   number of acts n = {n_acts}")
    print(f"   outward increment norm = {norm}")

    # Helper: Frobenius norm
    def fro_norm(mat: np.ndarray) -> float:
        return np.linalg.norm(mat, ord='fro')

    # 2.1) Ledger sector diagnostic
    def ledger_residual(alpha: float, m_dim: int = 64) -> float:
        """Compute the Frobenius norm of S†S + X†X − I for a given α.

        Parameters
        ----------
        alpha : float
            The allocation fraction (velocity fraction) α.
        m_dim : int
            The dimension of the diagonal matrices used to define S†S and X†X.

        Returns
        -------
        float
            The Frobenius norm of (S†S + X†X − I).
        """
        S2 = np.diag(np.full(m_dim, 1.0 - alpha ** 2))
        X2 = np.diag(np.full(m_dim, alpha ** 2))
        I = np.eye(m_dim)
        return fro_norm(S2 + X2 - I)

    ledger_gate = 1e-12  # strict threshold for ledger diagnostic
    ledger_res_by_alpha = {a: ledger_residual(a, m_dim=64) for a in alpha_grid}

    print("\n2.1) Ledger sector diagnostic: ||S†S + X†X − I||_F per α")
    for a, r in ledger_res_by_alpha.items():
        print(f"   α={a:>4}: residual={r:.3e}")
    ledger_green = max(ledger_res_by_alpha.values()) <= ledger_gate
    print(f"   Gate threshold: {ledger_gate:.1e}  → Ledger sector green? {ledger_green}")

    # 2.2) Per‑act bound typing for cones
    def per_act_b(c_val: float, Theta_val: float, kappa_val: float) -> float:
        return min(c_val * Theta_val, kappa_val)

    b_base = 1.0
    print("\n2.2) Per‑act bound typing:")
    print(f"   b_base (normalized) = {b_base}")
    print("   For gate‑variation runs, we use b(c,Θ,κ)=min(c*Θ, κ).")

    # No‑skip composition check (choose α=0.75 for demonstration)
    alpha_sample = 0.75
    dx_steps = np.full(n_acts, alpha_sample * b_base)
    long_move = dx_steps.sum()
    sum_of_neighbors = np.add.reduce(dx_steps)  # identical operation
    noskip_diff = abs(long_move - sum_of_neighbors)
    noskip_pass = np.isclose(noskip_diff, 0.0, atol=1e-12)
    print(f"   No‑skip composition check (α={alpha_sample}): |∑Δx_t − (long move)| = {noskip_diff:.3e} → pass? {noskip_pass}")

    # 2.3) Dilation simulation
    def simulate_worldline_dilation(alpha: float, n: int, dt_step: float):
        """Simulate a worldline for a given α and count measured dilation γ.

        Uses Δx_t = α * c * Δt and Δτ_t = sqrt((Δt)^2 − (Δx_t)^2 / c^2).
        
        Parameters
        ----------
        alpha : float
            The allocation fraction for each act.
        n : int
            The number of acts (worldline segments).
        dt_step : float
            The +1 time per act (usually 1 for normalized units).

        Returns
        -------
        tuple
            (dx, dtau, tau_total, gamma_meas):
              - dx : ndarray of outward displacements for each act
              - dtau : ndarray of proper-time ticks for each act
              - tau_total : total proper time over n acts
              - gamma_meas : measured dilation factor (n*dt_step / tau_total)
        """
        dx = np.full(n, alpha * c * dt_step)
        dtau = np.sqrt(np.maximum(0.0, dt_step**2 - (dx**2) / (c**2)))
        tau_total = dtau.sum()
        gamma_meas = (n * dt_step) / tau_total if tau_total > 0 else float('inf')
        return dx, dtau, tau_total, gamma_meas

    # Collect γ results for each α
    gamma_rows = []
    for a in alpha_grid:
        _, _, tau_total, gamma_meas = simulate_worldline_dilation(a, n_acts, dt)
        gamma_theory = 1.0 / math.sqrt(max(1e-300, 1.0 - a * a))
        rel_err = 0.0 if gamma_theory == 0 else abs(gamma_meas - gamma_theory) / gamma_theory
        gamma_rows.append({
            "alpha": a,
            "gamma_meas": gamma_meas,
            "gamma_theory": gamma_theory,
            "rel_error": rel_err,
        })

    gamma_df = pd.DataFrame(gamma_rows)

    # Fit γ_meas vs γ_theory with slope
    x = gamma_df["gamma_theory"].values
    y = gamma_df["gamma_meas"].values
    if np.dot(x, x) != 0:
        slope = float(np.dot(x, y) / np.dot(x, x))
    else:
        slope = float('nan')
    slope_error = abs(slope - 1.0)
    rmse_rel = math.sqrt(float(np.mean((gamma_df["rel_error"].values) ** 2)))
    max_rel_err = float(gamma_df["rel_error"].max())

    print("\n2.3) Dilation results (γ_meas vs γ_theory):")
    print(f"   slope (γ_meas ~ slope * γ_theory) = {slope:.12f}  → slope error = {slope_error:.3e}")
    print(f"   RMSE of relative error across α grid = {rmse_rel:.3e}")
    print(f"   Max relative error across α grid = {max_rel_err:.3e}")

    # 2.4) Cones
    def cone_envelope(alpha: float, count: int, b_val: float) -> np.ndarray:
        """Return the cumulative maximum reachability envelope for given α, n acts.
        Each act contributes α * b_val to the outward displacement.
        """
        return np.cumsum(np.full(count, alpha * b_val))

    N = np.arange(1, n_acts + 1)
    cone_baseline = {a: cone_envelope(a, n_acts, b_base) for a in alpha_grid}
    boundary_errors = []
    for a in alpha_grid:
        # Compose path using saturating dx for demonstration
        path = np.full(n_acts, a * b_base).cumsum()
        boundary = cone_baseline[a]
        if boundary[-1] != 0:
            err = abs(path[-1] - boundary[-1]) / boundary[-1]
        else:
            err = 0.0
        boundary_errors.append(err)
    cone_boundary_error = max(boundary_errors) if boundary_errors else 0.0
    print("\n2.4) Cones: boundary error on baseline (final step, worst α):", f"{cone_boundary_error:.3e}")

    # Gate variations demonstration for cones with α=0.9
    gate_variations = [
        {"Theta": 0.5, "kappa": 1.0},
        {"Theta": 1.0, "kappa": 1.0},
        {"Theta": 1.5, "kappa": 2.0},
    ]
    alpha_for_gate_variation = 0.9
    var_results = []
    for gv in gate_variations:
        b_var = per_act_b(c, gv["Theta"], gv["kappa"])
        env = cone_envelope(alpha_for_gate_variation, n_acts, b_var)
        var_results.append((gv, b_var, env))

    # 2.5) Re‑centering invariance
    shift = 137
    a_recenter = 0.75
    dx_orig, dtau_orig, tau_total_orig, gamma_meas_orig = simulate_worldline_dilation(a_recenter, n_acts, dt)
    dx_rot = np.roll(dx_orig, shift)
    dtau_rot = np.sqrt(np.maximum(0.0, dt**2 - (dx_rot**2) / (c**2)))
    tau_total_rot = dtau_rot.sum()
    gamma_meas_rot = (n_acts * dt) / tau_total_rot if tau_total_rot > 0 else float('inf')
    recenter_pass = (
        math.isclose(gamma_meas_orig, gamma_meas_rot, rel_tol=1e-12, abs_tol=1e-12)
        and math.isclose(dx_orig.sum(), dx_rot.sum(), rel_tol=1e-12, abs_tol=1e-12)
    )
    print("\n2.5) Re‑centering invariance (rotate by +137 acts):")
    print(f"   γ_meas (orig)={gamma_meas_orig:.12f}, γ_meas (rot)={gamma_meas_rot:.12f} → pass? {recenter_pass}")

    # 2.6) No‑skip audit
    print("\n2.6) No‑skip audit:", "passed" if noskip_pass else "FAILED")

    # 2.7) Optional homeostatic allocation run
    run_homeostatic = True
    eta = 0.05
    q_star = 0.9
    alpha0 = 0.2
    alpha_series = None
    gamma_meas_homeo = None
    cone_homeo = None
    if run_homeostatic:
        alpha_series = np.zeros(n_acts)
        alpha_series[0] = alpha0
        # Use simple approximation q_t ≈ alpha_t for homeostatic control
        for t in range(n_acts - 1):
            alpha_series[t + 1] = np.clip(alpha_series[t] + eta * (q_star - alpha_series[t]), 0.0, 0.999999)
        dx_homeo = alpha_series * c * dt
        dtau_homeo = np.sqrt(np.maximum(0.0, dt**2 - (dx_homeo**2) / (c**2)))
        tau_total_homeo = dtau_homeo.sum()
        gamma_meas_homeo = (n_acts * dt) / tau_total_homeo if tau_total_homeo > 0 else float('inf')
        cone_homeo = np.cumsum(alpha_series * b_base)

    # ----- Plotting -----
    # γ vs α plot
    plt.figure()
    plt.plot(gamma_df["alpha"].values, gamma_df["gamma_theory"].values, marker='o', label="γ_theory(α)")
    plt.plot(gamma_df["alpha"].values, gamma_df["gamma_meas"].values, marker='x', linestyle='--', label="γ_meas(α)")
    plt.xlabel("α")
    plt.ylabel("γ")
    plt.title("Time dilation: γ_meas(α) vs γ_theory(α)")
    plt.legend()
    gamma_plot_path = os.path.join(sr_out_dir, "sr_v2_gamma.png")
    plt.savefig(gamma_plot_path, bbox_inches='tight')
    plt.close()

    # Cone envelopes baseline + interior sample path
    plt.figure()
    for a in alpha_grid:
        plt.plot(N, cone_baseline[a], label=f"cone α={a}")
    # sample path inside the cone (choose α=0.8*0.75 to show interior)
    sample_path = np.cumsum(np.full(n_acts, 0.8 * 0.75 * b_base))
    plt.plot(N, sample_path, linestyle='--', label="sample path (inside)")
    plt.xlabel("act count N")
    plt.ylabel("X_max / composed X")
    plt.title("Cones (baseline b=1) & a sampled interior path")
    plt.legend()
    cones_baseline_path = os.path.join(sr_out_dir, "sr_v2_cones_baseline.png")
    plt.savefig(cones_baseline_path, bbox_inches='tight')
    plt.close()

    # Cone envelopes under gate variations
    plt.figure()
    for (gv, b_var, env) in var_results:
        label = f"Theta={gv['Theta']}, kappa={gv['kappa']}, b={b_var:.2f}"
        plt.plot(N, env, label=label)
    plt.xlabel("act count N")
    plt.ylabel("X_max")
    plt.title("Cone envelopes under gate variations (α=0.9)")
    plt.legend()
    cones_variations_path = os.path.join(sr_out_dir, "sr_v2_cones_variations.png")
    plt.savefig(cones_variations_path, bbox_inches='tight')
    plt.close()

    alpha_homeo_plot_path = None
    if run_homeostatic:
        plt.figure()
        plt.plot(np.arange(n_acts), alpha_series)
        plt.xlabel("act index t")
        plt.ylabel("α_t")
        plt.title("Homeostatic allocation time‑series (η=0.05, q*=0.9)")
        alpha_homeo_plot_path = os.path.join(sr_out_dir, "sr_v2_alpha_homeo.png")
        plt.savefig(alpha_homeo_plot_path, bbox_inches='tight')
        plt.close()

    # ----- Save manifest and tables -----
    # Determine acceptance metrics
    ledger_accept = ledger_green
    dilation_accept = (slope_error <= 0.03) and (max_rel_err <= 0.03)
    cones_accept = (cone_boundary_error <= 0.05)
    recenter_accept = recenter_pass
    noskip_accept = noskip_pass

    manifest = {
        "demo": "sr-v2",
        "seed": seed,
        "units": {"c": c},
        "gates": {"Theta": Theta_ladder, "kappa": kappa_ladder},
        "per_act_bound": {"b": b_base, "b_formula_for_variations": "b = min(c*Theta, kappa)"},
        "allocation": {"mode": "fixed-grid", "alpha_grid": alpha_grid},
        "acts": n_acts,
        "norm": norm,
        "checks": {
            "ledger_fro_residual_max": 1.0e-8,
            "dilation_slope_error_max": 0.03,
            "cone_boundary_error_max": 0.05,
            "recenter_shift": "+137",
        },
        "optional_homeostatic": {
            "enabled": run_homeostatic,
            "eta": eta,
            "q_star": q_star,
            "alpha0": alpha0,
        },
    }
    manifest_json_path = os.path.join(sr_out_dir, "sr_v2_manifest.json")
    manifest_yaml_path = os.path.join(sr_out_dir, "sr_v2_manifest.yaml")
    with open(manifest_json_path, "w") as f_json:
        json.dump(manifest, f_json, indent=2)
    with open(manifest_yaml_path, "w") as f_yaml:
        yaml.safe_dump(manifest, f_yaml, sort_keys=False)

    # Save gamma table
    gamma_csv_path = os.path.join(sr_out_dir, "sr_v2_gamma_table.csv")
    gamma_df.to_csv(gamma_csv_path, index=False)

    # Save text report summarizing acceptance metrics
    report_lines = []
    report_lines.append("Governance checklist:")
    for item in governance_checklist:
        report_lines.append(" - " + item)
    report_lines.append("")
    report_lines.append(f"Ledger residuals (Frobenius): { {a: float(r) for a, r in ledger_res_by_alpha.items()} }")
    report_lines.append(f"Ledger threshold: {ledger_gate:.1e}, green: {ledger_green}")
    report_lines.append(
        f"Dilation slope vs theory: slope={slope:.12f}, slope_error={slope_error:.3e}, "
        f"RMSE_rel={rmse_rel:.3e}, max_rel_err={max_rel_err:.3e}, accept={dilation_accept}"
    )
    report_lines.append(f"Cone boundary error: {cone_boundary_error:.3e}, accept={cones_accept}")
    report_lines.append(f"Re‑centering (+137) invariance: {recenter_accept}")
    report_lines.append(f"No‑skip factorization: {noskip_accept}")
    if run_homeostatic and gamma_meas_homeo is not None:
        report_lines.append(f"Homeostatic run γ_meas={gamma_meas_homeo:.6f}")
    report_text = "\n".join(report_lines)
    report_path = os.path.join(sr_out_dir, "sr_v2_report.txt")
    with open(report_path, "w") as f_report:
        f_report.write(report_text)

    # Print final acceptance metrics summary to console
    print("\n3) Acceptance metrics:")
    print(f"   Ledger sector green (≤1e-8): {ledger_accept}")
    print(
        f"   Dilation slope error ≤ 3%? {dilation_accept}  (slope_error={slope_error:.3e}, "
        f"RMSE_rel={rmse_rel:.3e}, max_rel_err={max_rel_err:.3e})"
    )
    print(
        f"   Cones boundary error ≤ 5%? {cones_accept}  (boundary_error={cone_boundary_error:.3e})"
    )
    print(f"   Re‑centering invariance pass? {recenter_accept}")
    print(f"   No‑skip factorization pass? {noskip_accept}")
    if run_homeostatic and gamma_meas_homeo is not None:
        print(f"   Homeostatic run γ_meas={gamma_meas_homeo:.6f}")

    # List produced files
    print("\n6) Files produced:")
    print("   γ plot:", gamma_plot_path)
    print("   Cones (baseline) plot:", cones_baseline_path)
    print("   Cones (gate variations) plot:", cones_variations_path)
    if alpha_homeo_plot_path:
        print("   Homeostatic α_t plot:", alpha_homeo_plot_path)
    print("   Report:", report_path)
    print("   Manifest (JSON):", manifest_json_path)
    print("   Manifest (YAML):", manifest_yaml_path)
    print("   γ table (CSV):", gamma_csv_path)


if __name__ == "__main__":
    # Run the baseline with outputs directed to ./results
    default_out_dir = os.path.join(os.path.dirname(__file__), os.pardir, "results")
    # Normalize path for writing results
    out_dir = os.path.abspath(default_out_dir)
    run_baseline(out_dir, seed=20251020)